/**
 * 
 */
package gov.va.med.mhv.usermgmt.service;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import gov.va.med.mhv.core.util.DescriptionBuilder;
import gov.va.med.mhv.usermgmt.enumeration.AccessDomain;
import gov.va.med.mhv.usermgmt.enumeration.AccessPermission;
import gov.va.med.mhv.usermgmt.enumeration.BloodTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.ContactMethodEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.CountryEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.GenderEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.MaritalStatusEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.PasswordHintQuestionEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.StateEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.TitleEnumeration;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.transfer.AccessControl;
import gov.va.med.mhv.usermgmt.transfer.AccessFeature;
import gov.va.med.mhv.usermgmt.transfer.AccessRole;
import gov.va.med.mhv.usermgmt.transfer.AccessRoleAccessControl;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.transfer.UserProfileAccessControl;
import gov.va.med.mhv.usermgmt.transfer.UserProfileAccessRole;
import gov.va.med.mhv.usermgmt.util.AccessControlUtils;
import gov.va.med.mhv.usermgmt.util.AccessDomainUtils;
import gov.va.med.mhv.usermgmt.util.AccessPermissionUtils;

/**
 * @author Rob Proper
 *
 */
public class TestAccessControlService extends BaseTestCase {
	
	private static final Log LOG = LogFactory.getLog(TestAccessControlService.
        class);
    
    public static final String PROPERTY = 
        "testAccessControlService.patientUserName";

    private static final String[] TEST_ROLE_NAME = { 
        "Test Role 1",
        "Test Role 2",
        "Test Role 3",
        "Test Role 4"
    };  

    private static final String[] TEST_FEATURE_VALUE = {
        "Test Feature 1",
        "Test Feature 2",
        "Test Feature 3",
        "Test Feature 4"
    };
    
    private static final AccessPermission[][] TEST_PERMISSIONS = {
        { AccessPermissionUtils.READ },
        { AccessPermissionUtils.READ, AccessPermissionUtils.UPDATE },
        { AccessPermissionUtils.DISABLED }
    };
    private final Long[] TEST_PERMISSION_MASKS = {
        AccessPermissionUtils.createMask(TEST_PERMISSIONS[0]),
        AccessPermissionUtils.createMask(TEST_PERMISSIONS[1]),
        AccessPermissionUtils.createMask(TEST_PERMISSIONS[2])
    };


    private String patientUserName = null; 
    
	private UserBuilder user = null;
    private Collection<AccessRole> roles = new ArrayList<AccessRole>();
    private Collection<AccessFeature> features = new ArrayList<AccessFeature>();
    private Collection<UserProfileAccessRole> userProfileRoles = 
        new ArrayList<UserProfileAccessRole>();
    private Collection<AccessControlChecker> accessControlCheckers =
        new ArrayList<AccessControlChecker>();

	/**
	 * Constructor for UserServiceAuthenticateTest.
	 * @param arg0
	 */
	public TestAccessControlService(String arg0) {
		super(arg0);
	}

    
	/*
	 * @see TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {
        for (AccessControlChecker checker: accessControlCheckers) {
            checker.tearDown();
        }
        accessControlCheckers.clear();
        for (UserProfileAccessRole role: userProfileRoles) {
            delete(role);
        }
        userProfileRoles.clear();
        for (AccessRole role: roles) {
            delete(role);
        }
        roles.clear();
        for (AccessFeature feature: features) {
            delete(feature);
        }
        features.clear();
		if (user != null) { 
            if (patientUserName == null) {
                user.tearDown();
            } // else: Do not delete staged patient user profile
		}
	}
	
    public void testOneRoleBasedAccessControls()  throws Exception {
        setUpTestUser();
        AccessFeature feature = addAccessFeature(AccessDomainUtils.PHR, 
            TEST_FEATURE_VALUE[0]);
        AccessRole role = addAccessRole(TEST_ROLE_NAME[0]);
        addUserProfileAccessRole(role, user.getUserProfile());
        addAccessRoleAccessControl(role, feature, TEST_PERMISSION_MASKS[0], 
            true);

        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        assertAccessControlCollection(response);
	}

    public void testTwoRoleBasedAccessControls() throws Exception {
        setUpTestUser();
        AccessFeature[] feature = {
            addAccessFeature(AccessDomainUtils.PHR, TEST_FEATURE_VALUE[0]),
            addAccessFeature(AccessDomainUtils.PHR, TEST_FEATURE_VALUE[1])
        };
        AccessRole role = addAccessRole(TEST_ROLE_NAME[0]);
        addUserProfileAccessRole(role, user.getUserProfile());
        addAccessRoleAccessControl(role, feature[0], TEST_PERMISSION_MASKS[0], 
            true);
        addAccessRoleAccessControl(role, feature[1], TEST_PERMISSION_MASKS[0],
            true);

        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        assertAccessControlCollection(response);
    }

    public void testTwoRoleBasedAccessControlsWithMoreRolesDefined() 
        throws Exception 
    {
        setUpTestUser();
        AccessFeature[] feature = {
            addAccessFeature(AccessDomainUtils.PHR, TEST_FEATURE_VALUE[0]),
            addAccessFeature(AccessDomainUtils.PHR, TEST_FEATURE_VALUE[1]),
            addAccessFeature(AccessDomainUtils.PHR, TEST_FEATURE_VALUE[2]),
        };
        AccessRole[] role = {
            addAccessRole(TEST_ROLE_NAME[0]),
            addAccessRole(TEST_ROLE_NAME[1])
        };
        addUserProfileAccessRole(role[0], user.getUserProfile());
    
        addAccessRoleAccessControl(role[0], feature[0], 
            TEST_PERMISSION_MASKS[0], true);
        addAccessRoleAccessControl(role[0], feature[1], 
            TEST_PERMISSION_MASKS[0], true);
        addAccessRoleAccessControl(role[1], feature[2], 
            TEST_PERMISSION_MASKS[0], false);
    
        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        
        assertAccessControlCollection(response);
    }

    public void testDefaultPatientAccessControlWithNonPatient() 
        throws Exception 
    {
        setUpTestUser();
        AccessFeature feature = addAccessFeature(AccessDomainUtils.PHR, 
            TEST_FEATURE_VALUE[0]);
        AccessRole role = addAccessRole(TEST_ROLE_NAME[0], true);
        addAccessRoleAccessControl(role, feature, TEST_PERMISSION_MASKS[0], 
            false);

        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        assertAccessControlCollection(response);
    }

    public void testDefaultPatientAccessControlWithPatient() 
        throws Exception 
    {
        if (!hasTestPatient("testDefaultPatientAccessControlWithPatient")) {
            return;
        }
        setUpTestUser(true);
        AccessFeature feature = addAccessFeature(AccessDomainUtils.PHR, 
            TEST_FEATURE_VALUE[0]);
        AccessRole role = addAccessRole(TEST_ROLE_NAME[0], true);
        addAccessRoleAccessControl(role, feature, TEST_PERMISSION_MASKS[0], 
            true);
    
        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        assertAccessControlCollection(response);
    }

    public void testOneUserBasedAccessControls()  throws Exception {
        setUpTestUser();
        AccessFeature feature = addAccessFeature(AccessDomainUtils.PHR, 
            TEST_FEATURE_VALUE[0]);
        AccessRole role = addAccessRole(TEST_ROLE_NAME[0]);
        addUserProfileAccessRole(role, user.getUserProfile());
        addUserProfileAccessControl(user.getUserProfile(), feature, 
            TEST_PERMISSION_MASKS[0], true);

        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        assertAccessControlCollection(response);
    }

    public void testTwoUserBasedAccessControls() throws Exception {
        setUpTestUser();
        AccessFeature[] feature = {
            addAccessFeature(AccessDomainUtils.PHR, TEST_FEATURE_VALUE[0]),
            addAccessFeature(AccessDomainUtils.PHR, TEST_FEATURE_VALUE[1])
        };
        AccessRole role = addAccessRole(TEST_ROLE_NAME[0]);
        addUserProfileAccessRole(role, user.getUserProfile());
        addUserProfileAccessControl(user.getUserProfile(), feature[0], 
            TEST_PERMISSION_MASKS[0], true);
        addUserProfileAccessControl(user.getUserProfile(), feature[1], 
            TEST_PERMISSION_MASKS[0], true);

        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        assertAccessControlCollection(response);
    }

    public void testRoleAndUserBasedAccessControls()  throws Exception {
        setUpTestUser();
        AccessFeature feature = addAccessFeature(AccessDomainUtils.PHR, 
            TEST_FEATURE_VALUE[0]);
        AccessRole role = addAccessRole(TEST_ROLE_NAME[0]);
        addUserProfileAccessRole(role, user.getUserProfile());
        addUserProfileAccessControl(user.getUserProfile(), feature, 
            TEST_PERMISSION_MASKS[0], true);
        addAccessRoleAccessControl(role, feature, 
            TEST_PERMISSION_MASKS[1], false);

        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        assertAccessControlCollection(response);
    }

    public void testMultipleRoleAndUserBasedAccessControls() throws Exception {
        setUpTestUser();
        AccessFeature feature = addAccessFeature(AccessDomainUtils.PHR, 
            TEST_FEATURE_VALUE[0]);
        AccessRole[] role = {
            addAccessRole(TEST_ROLE_NAME[0]),
            addAccessRole(TEST_ROLE_NAME[1], true)
        };
        addUserProfileAccessRole(role[0], user.getUserProfile());
        addUserProfileAccessControl(user.getUserProfile(), feature, 
            TEST_PERMISSION_MASKS[0], true);
        addAccessRoleAccessControl(role[0], feature, 
            TEST_PERMISSION_MASKS[1], false);
        addAccessRoleAccessControl(role[1], feature, 
            TEST_PERMISSION_MASKS[1], false);

        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        assertAccessControlCollection(response);
    }

    @SuppressWarnings("unchecked")
    public void testFeatureDisabledForUser() throws Exception {
        setUpTestUser();
        AccessFeature feature = addAccessFeature(AccessDomainUtils.PHR, 
            TEST_FEATURE_VALUE[0]);
        AccessRole[] role = {
            addAccessRole(TEST_ROLE_NAME[0]),
            addAccessRole(TEST_ROLE_NAME[1], true)
        };
        addUserProfileAccessRole(role[0], user.getUserProfile());
        addUserProfileAccessControl(user.getUserProfile(), feature, 
            TEST_PERMISSION_MASKS[2], true);
        addAccessRoleAccessControl(role[1], feature, TEST_PERMISSION_MASKS[0], 
            false);

        AccessControlCollectionServiceResponse response = 
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(user.getUserName());
        assertAccessControlCollection(response);
        assertFalse(AccessControlUtils.hasAccess(feature, 
            AccessPermissionUtils.READ, response.getAccessControls()));
        assertFalse(AccessControlUtils.hasAccess(feature.getDomain(),
            feature.getName(), AccessPermissionUtils.READ, 
            response.getAccessControls()));
    }

  

    private void setUpTestUser() throws Exception {
        setUpTestUser(false);
    }
    
	private void setUpTestUser(boolean isPatient) throws Exception {
        if (isPatient) {
            // Use staged patient user profile
            UserProfileServiceResponse response = ServiceDelegateFactory.
                createUserProfileServiceDelegate().getProfileForUser(
                patientUserName);
            assertNoErrors(response);
            assertNotNull("Patient with userName '" + patientUserName 
                + "' not found", response.getUserProfile());
            user = new UserBuilder(response.getUserProfile(), "Tester1&");
        } else {
            // Create a new user profile
            user = new UserBuilder(createUserProfile(null, "John", "Doe", 
                UserBuilder.createDate(Calendar.JANUARY, 1, 1970)), 
                "%password0");
            assertNotNull("Failed to set up test user", user);
            user.setUp();
        }
	}

    private static UserProfile createUserProfile(String userName, 
        String firstName, String lastName, Date birthDate) 
    {
        UserProfile userProfile = TransferObjectFactory.createUserProfile();
        userProfile.setUserName(userName);
        userProfile.setFirstName(firstName);
        userProfile.setLastName(lastName);
        userProfile.setMiddleName("");
        userProfile.setBirthDate(birthDate);
        userProfile.setTitle(TitleEnumeration.getEnum(TitleEnumeration.MR));
        userProfile.setBloodType(BloodTypeEnumeration.getEnum(
            BloodTypeEnumeration.O_NEGATIVE));
        userProfile.setIsOrganDonor(Boolean.TRUE);
        userProfile.setGender(GenderEnumeration.getEnum(GenderEnumeration.MALE));
        userProfile.setMaritalStatus(MaritalStatusEnumeration.getEnum(
            MaritalStatusEnumeration.DIVORCED));
        userProfile.setIsVeteran(Boolean.TRUE);
        userProfile.setIsPatient(false);
        userProfile.setAddressStreet1("1 Some Street");
        userProfile.setAddressCity("Nowhere Town");
        userProfile.setAddressState(StateEnumeration.getEnum(StateEnumeration.VA));
        userProfile.setAddressPostalCode("12345");
        userProfile.setAddressCountry(CountryEnumeration.getEnum(
            CountryEnumeration.UNITED_STATES));
        userProfile.setContactInfoContactMethod(ContactMethodEnumeration.getEnum(
            ContactMethodEnumeration.HOMEPHONE));
        userProfile.setContactInfoHomePhone("555-555-5555");
        userProfile.setContactInfoEmail(firstName + "." + lastName + 
            "serv.domain.ext");
        userProfile.setPasswordHintQuestion1(PasswordHintQuestionEnumeration.
            getEnum(PasswordHintQuestionEnumeration.FAVORITEFOOD));
        userProfile.setPasswordHintAnswer1("Pizza");
        userProfile.setPasswordHintQuestion2(PasswordHintQuestionEnumeration.
            getEnum(PasswordHintQuestionEnumeration.PETNAME));
        userProfile.setPasswordHintAnswer2("Spot");
        userProfile.setAcceptDisclaimer(Boolean.TRUE);
        userProfile.setAcceptPrivacy(Boolean.TRUE);
        userProfile.setAcceptTerms(Boolean.TRUE);
        return userProfile;
    }
    
    private boolean hasTestPatient(String testName) {
        patientUserName = System.getProperty(PROPERTY);
        boolean hasTestUser = !StringUtils.isBlank(patientUserName);
        if (!hasTestUser) {
            LOG.warn("Skipping " + testName 
                + ", because no test user name specified in property '" 
                + PROPERTY + "'.");
        }
        return hasTestUser;
    }

    private AccessRole addAccessRole(String roleName) throws Exception {
        return addAccessRole(roleName, false);
    }

    private AccessRole addAccessRole(String roleName, 
        boolean isDefaultpatientRole) throws Exception 
    {
        AccessRole role = TransferObjectFactory.createAccessRole();
        role.setName(roleName);
        role.setIsDefaultPatientRole(isDefaultpatientRole);
        AccessRoleServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().save(role);
        assertNoErrors(response);
        role = response.getAccessRole(); 
        assertNotNull(role);
        roles.add(role);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Created " + DescriptionBuilder.describe(role));
        }
        return role;
    }

    private void delete(AccessRole accessRole) throws Exception {
        if ((accessRole == null) || (accessRole.getId() == null)) {
            return;
        }
        AccessRoleServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().delete(accessRole.
             getAccessRolePK());
        if (hasErrors(response)) {
            logMessages(response, AccessRole.class, LOG);
        }
    }

    private AccessFeature addAccessFeature(AccessDomain domain, String name) 
        throws Exception 
    {
        AccessFeature feature = TransferObjectFactory.createAccessFeature();
        feature.setDomain(domain);
        feature.setName(name);
        AccessFeatureServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().save(feature);
        assertNoErrors(response);
        feature = response.getAccessFeature(); 
        assertNotNull(feature);
        features.add(feature);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Created " + DescriptionBuilder.describe(feature));
        }
        return feature;
    }

    private void delete(AccessFeature accessFeature) throws Exception {
        if ((accessFeature == null) || (accessFeature.getId() == null)) {
            return;
        }
        AccessFeatureServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().delete(accessFeature.
             getAccessFeaturePK());
        if (hasErrors(response)) {
            logMessages(response, AccessFeature.class, LOG);
        }
    }

    private UserProfileAccessRole addUserProfileAccessRole(AccessRole role, 
        UserProfile userProfile) throws Exception 
    {
        UserProfileAccessRole userProfileRole = TransferObjectFactory.
            createUserProfileAccessRole();
        userProfileRole.setRole(role);
        userProfileRole.setUserProfile(userProfile);
        UserProfileAccessRoleServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().save(
            userProfileRole);
        assertNoErrors(response);
        userProfileRole = response.getUserProfileAccessRole(); 
        assertNotNull(userProfileRole);
        userProfileRoles.add(userProfileRole);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Created " + DescriptionBuilder.describe(userProfileRole));
        }
        return userProfileRole;
    }

    private void delete(UserProfileAccessRole accessRole) throws Exception {
        if ((accessRole == null) || (accessRole.getId() == null)) {
            return;
        }
        UserProfileAccessRoleServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().delete(accessRole.
             getUserProfileAccessRolePK());
        if (hasErrors(response)) {
            logMessages(response, UserProfileAccessRole.class, LOG);
        }
    }

    @SuppressWarnings("unchecked")
    private void assertAccessControlCollection(
        AccessControlCollectionServiceResponse response) 
    {
        assertNotNull(response);
        Collection<AccessControl> accessControls = response.getAccessControls(); 
        assertNotNull(accessControls);
        if (accessControlCheckers != null) {
            for (AccessControlChecker checker: accessControlCheckers) {
                checker.assertControl(accessControls);
            }
        }
            
    }

    private void addUserProfileAccessControl(UserProfile userProfile,
        AccessFeature feature, Long permissions, boolean hasControl) 
        throws Exception 
    {
        assertNotNull(userProfile);
        assertNotNull(feature);
        assertNotNull(permissions);
        accessControlCheckers.add(new UserProfileAccessControlChecker(
            userProfile, feature, permissions, hasControl));
    }

    private void addAccessRoleAccessControl(AccessRole role,
        AccessFeature feature, Long permissions, boolean hasControl) 
        throws Exception 
    {
        assertNotNull(role);
        assertNotNull(feature);
        assertNotNull(permissions);
        accessControlCheckers.add(new AccessRoleAccessControlChecker(role,
            feature, permissions, hasControl));
    }



    private static abstract class AccessControlChecker {
        private final boolean hasControl;
        protected AccessControlChecker(boolean hasControl) {
            this.hasControl = hasControl;
        }
        public abstract void tearDown() throws Exception;
        public abstract String describe();

        public void assertControl(Collection<AccessControl> controls) {
            if (hasControl) {
                assertNotNull(controls);
                assertFalse(controls.isEmpty());
            }
            if (controls != null) {
                for (AccessControl accessControl: controls) {
                    if (isControl(accessControl)) {
                        assertTrue("Not expected to find control " + describe(),
                            hasControl);
                        return;
                    }
                }
            }
            assertFalse("Expected to find control " + describe(), hasControl);
        }
        protected abstract boolean isControl(AccessControl accessControl);

        protected boolean hasControl() {
            return hasControl;
        }
    }

    private static final class AccessRoleAccessControlChecker 
        extends AccessControlChecker
    {
        private final AccessRoleAccessControl control;
        
        public AccessRoleAccessControlChecker(AccessRole role,
            AccessFeature feature, Long permissions, boolean hasControl) 
        {
            super(hasControl);
            AccessRoleAccessControl accessControl = TransferObjectFactory.
                createAccessRoleAccessControl();
            accessControl.setFeature(feature);
            accessControl.setRole(role);
            accessControl.setPermissions(permissions);
            AccessRoleAccessControlServiceResponse response = 
                ServiceDelegateFactory.createEntityMaintenanceServiceDelegate().
                save(accessControl);
            assertNoErrors(response);
            accessControl = response.getAccessRoleAccessControl(); 
            assertNotNull(accessControl);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Created " + AccessControlUtils.describe(
                    accessControl));
            }
            
            this.control = accessControl;
        }

        /* (non-Javadoc)
         * @see gov.va.med.mhv.usermgmt.service.TestAccessControlService.
         * AccessControlChecker#tearDown()
         */
        @Override
        public void tearDown() throws Exception {
            AccessRoleAccessControlServiceResponse response = 
                ServiceDelegateFactory.createEntityMaintenanceServiceDelegate().
                delete(control.getAccessRoleAccessControlPK());
            if (hasErrors(response)) {
                logMessages(response, AccessRoleAccessControl.class, LOG);
            }
        }

        /* (non-Javadoc)
         * @see gov.va.med.mhv.usermgmt.service.TestAccessControlService.
         * AccessControlChecker#equals(
         * gov.va.med.mhv.usermgmt.transfer.AccessControl)
         */
        @Override
        protected boolean isControl(AccessControl accessControl) {
            return AccessControlUtils.hasAccess(control.getFeature(), 
                       control.getPermissions(), accessControl)
                && (accessControl.getRoleId() != null)
                && AccessControlUtils.isForRole(control.getRole(), 
                        accessControl);
            
        }

        /* (non-Javadoc)
         * @see gov.va.med.mhv.usermgmt.service.TestAccessControlService.
         * AccessControlChecker#describe()
         */
        @Override
        public String describe() {
            return AccessControlUtils.describe(control);        
        }
    }

    private static final class UserProfileAccessControlChecker 
        extends AccessControlChecker    
    {
        private final UserProfileAccessControl control;
        
        public UserProfileAccessControlChecker(UserProfile userProfile, 
            AccessFeature feature, Long permissions, boolean hasControl) 
        {
            super(hasControl);
            UserProfileAccessControl accessControl = TransferObjectFactory.
                createUserProfileAccessControl();
            accessControl.setFeature(feature);
            accessControl.setUserProfile(userProfile);
            accessControl.setPermissions(permissions);
            UserProfileAccessControlServiceResponse response = 
                ServiceDelegateFactory.createEntityMaintenanceServiceDelegate().
                save(accessControl);
            assertNoErrors(response);
            accessControl = response.getUserProfileAccessControl(); 
            assertNotNull(accessControl);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Created " + DescriptionBuilder.describe(
                    accessControl));
            }
            this.control = accessControl;
        }
    
        /* (non-Javadoc)
         * @see gov.va.med.mhv.usermgmt.service.TestAccessControlService.
         * AccessControlChecker#tearDown()
         */
        @Override
        public void tearDown() throws Exception {
            UserProfileAccessControlServiceResponse response = 
                ServiceDelegateFactory.createEntityMaintenanceServiceDelegate().
                delete(control.getUserProfileAccessControlPK());
            if (hasErrors(response)) {
                logMessages(response, UserProfileAccessControl.class, LOG);
            }
        }
    
        /* (non-Javadoc)
         * @see gov.va.med.mhv.usermgmt.service.TestAccessControlService.
         * AccessControlChecker#equals(gov.va.med.mhv.usermgmt.transfer.
         * AccessControl)
         */
        @Override
        protected boolean isControl(AccessControl accessControl) {
            return AccessControlUtils.hasAccess(control.getFeature(), 
                       control.getPermissions(), accessControl)
                && (accessControl.getUserProfileId() != null)
                && AccessControlUtils.isForUserProfile(control.
                       getUserProfile(), accessControl);
        }

        /* (non-Javadoc)
         * @see gov.va.med.mhv.usermgmt.service.TestAccessControlService.
         * AccessControlChecker#describe()
         */
        @Override
        public String describe() {
            return AccessControlUtils.describe(control);        
        }
    }
}
